from myapp.my_class import MyClass
from myapp.config import Config
import os
from unittest.mock import patch, Mock, mock_open
import pytest
from unittest import TestCase


@pytest.fixture
def filename():
    return "filename.md"

@pytest.fixture
def mocked_config(filename):
    # Prepare the Config object to inject
    config = Mock()
    config.__class__ = Config
    mocked_get_param = Mock()
    mocked_get_param.return_value = filename
    config.get_param = mocked_get_param

    return config


def test_init():
    config = Mock()
    instance = MyClass(config)

    assert isinstance(instance, MyClass)
    assert instance.config == config

def test_get_content_raise_when_no_file(mocked_config, filename):
    # Prepare the output of the os module
    mocked_os_path_exist = Mock()
    mocked_os_path_exist.return_value = False

    # Instantiate
    instance = MyClass(mocked_config)

    # Patch what should not be executed
    with patch.object(os.path, "exists", new=mocked_os_path_exist):
        
        # This test execution expects to raise an error
        with TestCase.assertRaises(instance, RuntimeError):
            content = instance.get_content()
    
    # Asserting now that we used the expected functions
    mocked_config.get_param.assert_called_once_with("filename")
    mocked_os_path_exist.assert_called_once_with(filename)

@patch("builtins.open", mock_open(read_data="I am content"))
def test_get_content_success(mocked_config, filename):
    # Prepare the output of the os module
    mocked_os_path_exist = Mock()
    mocked_os_path_exist.return_value = True

    # Instantiate
    instance = MyClass(mocked_config)

    # Patch what should not be executed
    with patch.object(os.path, "exists", new=mocked_os_path_exist):
        content = instance.get_content()

    # Main assert
    assert content == "I am content"
    
    # Asserting now that we used the expected functions
    mocked_config.get_param.assert_called_once_with("filename")
    mocked_os_path_exist.assert_called_once_with(filename)

@pytest.mark.parametrize(
    argnames=('path_exists', 'expected_result'),
    argvalues=[
        (False, False),
        (True, "I am content"),
    ],
)
@patch("builtins.open", mock_open(read_data="I am content"))
def test_get_content(mocked_config, filename, path_exists, expected_result):
    # Prepare the output of the os module
    mocked_os_path_exist = Mock()
    mocked_os_path_exist.return_value = path_exists

    # Instantiate
    instance = MyClass(mocked_config)

    # Patch what should not be executed
    with patch.object(os.path, "exists", new=mocked_os_path_exist):
        
        if isinstance(expected_result, bool) and expected_result is False:
            with TestCase.assertRaises(instance, RuntimeError):
                content = instance.get_content()
        else:
            content = instance.get_content()
            
            assert content == expected_result
    
    # Asserting now that we used the expected functions
    mocked_config.get_param.assert_called_once_with("filename")
    mocked_os_path_exist.assert_called_once_with(filename)